/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.firebase.internal

import android.content.Context
import com.google.android.gms.tasks.Task
import com.google.firebase.messaging.FirebaseMessaging
import com.moengage.core.LogLevel
import com.moengage.core.internal.SdkInstanceManager
import com.moengage.core.internal.global.GlobalState
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.pushbase.internal.ATTRIBUTE_TOKEN_REGISTERED_MOE
import com.moengage.pushbase.internal.getRetryInterval
import com.moengage.pushbase.internal.notifyTokenAvailable
import com.moengage.pushbase.model.PushService
import java.util.concurrent.Executors
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.TimeUnit

/**
 * @author Umang Chamaria
 */

internal object TokenRegistrationHandler {

    private val tag = "${MODULE_TAG}TokenRegistrationHandler"

    private var scheduler: ScheduledExecutorService? = null

    fun registerForPush(context: Context) {
        try {
            Logger.print { "$tag registerForPush() : Will try to register for push" }
            if (!shouldRegisterForPush(SdkInstanceManager.getAllInstances())) return
            FirebaseMessaging.getInstance().token.addOnCompleteListener { task ->
                try {
                    processTokenRegistrationResult(task, context)
                } catch (t: Throwable) {
                    Logger.print(LogLevel.ERROR, t) { "$tag registerForPush() : " }
                    scheduleTokenRegistrationRetry(context)
                }
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag registerForPush() : " }
        }
    }

    private fun processTokenRegistrationResult(task: Task<String>, context: Context) {
        if (!task.isSuccessful) {
            Logger.print(
                LogLevel.ERROR,
                task.exception
            ) { "$tag processTokenRegistrationResult() : Task<InstanceIdResult> failed. " }
            scheduleTokenRegistrationRetry(context)
            return
        }
        val token = task.result
        if (token.isNullOrBlank()) {
            scheduleTokenRegistrationRetry(context)
            return
        }
        processPushToken(context, token)
    }

    fun processPushToken(context: Context, pushToken: String) {
        Logger.print { "$tag processPushToken() : Token: $pushToken" }
        val token = ripMultiplexingExtras(pushToken)
        notifyTokenAvailable(token, PushService.FCM, FcmCache.tokenListeners)
        for (instance in SdkInstanceManager.getAllInstances().values) {
            if (instance.initConfig.push.fcm.isRegistrationEnabled) {
                FcmInstanceProvider.getControllerForInstance(instance)
                    .processPushToken(context, token, ATTRIBUTE_TOKEN_REGISTERED_MOE)
            }
        }
    }

    private fun scheduleTokenRegistrationRetry(context: Context) {
        if (!GlobalState.isForeground) return
        Logger.print { "$tag scheduleTokenRegistrationRetry() : " }
        if (scheduler == null || scheduler?.isShutdown == true) {
            scheduler = Executors.newScheduledThreadPool(1)
        }
        val runnable = Runnable {
            Logger.print { "$tag run() : Will attempt to register for token" }
            registerForPush(context)
        }
        scheduler?.schedule(
            runnable,
            getRetryInterval(SdkInstanceManager.getAllInstances()),
            TimeUnit.SECONDS
        )
    }

    fun onAppBackground(context: Context) {
        try {
            Logger.print { "$tag onAppBackground() : " }
            if (scheduler != null && (scheduler?.isShutdown == false)) {
                scheduler?.shutdownNow()
            }
        } catch (t: Throwable) {
            Logger.print(LogLevel.ERROR, t) { "$tag onAppBackground() : " }
        }
    }

    private fun shouldRegisterForPush(sdkInstances: Map<String, SdkInstance>): Boolean {
        for (instance in sdkInstances.values) {
            if (instance.initConfig.push.fcm.isRegistrationEnabled) return true
        }
        return false
    }
}