/*
 *
 *  * Copyright (c) 2020-2021, Java知识图谱 (http://www.altitude.xin).
 *  *
 *  * Licensed under the Apache License, Version 2.0 (the "License");
 *  * you may not use this file except in compliance with the License.
 *  * You may obtain a copy of the License at
 *  *
 *  *     http://www.apache.org/licenses/LICENSE-2.0
 *  *
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS,
 *  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  * See the License for the specific language governing permissions and
 *  * limitations under the License.
 *
 */

package xin.altitude.common.service.impl;

import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import xin.altitude.common.util.EntityUtils;

import java.io.Serializable;
import java.util.Collection;
import java.util.List;

import static xin.altitude.common.util.BeanCopyUtils.copyProperties;
import static xin.altitude.common.util.EntityUtils.convertList;

/**
 * 查询直接返回实体VO
 *
 * @author explore
 * @since 2021/11/29 17:57
 **/
public class JoinServiceImpl<M extends BaseMapper<T>, T> extends ServiceImpl<M, T> {
    /**
     * 通过主键查询Vo
     *
     * @param id    主键ID
     * @param clazz 实体Vo类实例对象
     * @param <VO>  实体Vo泛型
     * @return 实体Vo对象
     */
    public <VO extends T> VO getVoById(Serializable id, Class<VO> clazz) {
        T model = super.getById(id);
        return copyProperties(model, clazz);
    }
    
    /**
     * 主键查询实体Vo集合
     *
     * @param ids   主键ID集
     * @param clazz 实体Vo类实例对象
     * @param <VO>  实体Vo泛型
     * @return 实体Vo对象集合
     */
    public <VO extends T> List<VO> listVoByIds(Collection<? extends Serializable> ids, Class<VO> clazz) {
        List<T> models = super.listByIds(ids);
        return convertList(models, e -> copyProperties(e, clazz));
    }
    
    /**
     * 分页查询实体对应的Vo
     *
     * @param page         分页对象实例
     * @param queryWrapper 查询条件
     * @param clazz        实体Vo
     * @param <E>          实体泛型
     * @param <VO>         实体Vo泛型
     * @return 实体Vo分页对象
     */
    public <E extends IPage<T>, VO extends T> IPage<VO> pageVo(E page, Wrapper<T> queryWrapper, Class<VO> clazz) {
        E modelPage = page(page, queryWrapper);
        return EntityUtils.convertPage2(modelPage, e -> copyProperties(e, clazz));
    }
    
    /**
     * 分页查询实体对应的Vo
     *
     * @param page  分页对象实例
     * @param clazz 实体Vo
     * @param <E>   实体泛型
     * @param <VO>  实体Vo泛型
     * @return 实体Vo分页对象
     */
    public <E extends IPage<T>, VO extends T> IPage<VO> pageVo(E page, Class<VO> clazz) {
        return pageVo(page, Wrappers.emptyWrapper(), clazz);
    }
}
