/*
 *
 *  * Copyright (c) 2020-2021, Java知识图谱 (http://www.altitude.xin).
 *  *
 *  * Licensed under the Apache License, Version 2.0 (the "License");
 *  * you may not use this file except in compliance with the License.
 *  * You may obtain a copy of the License at
 *  *
 *  *     http://www.apache.org/licenses/LICENSE-2.0
 *  *
 *  * Unless required by applicable law or agreed to in writing, software
 *  * distributed under the License is distributed on an "AS IS" BASIS,
 *  * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  * See the License for the specific language governing permissions and
 *  * limitations under the License.
 *
 */

package xin.altitude.common.service.impl;

import com.baomidou.mybatisplus.core.mapper.BaseMapper;
import com.baomidou.mybatisplus.extension.activerecord.Model;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import xin.altitude.common.model.ModelWrapper;

import javax.annotation.PostConstruct;
import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static xin.altitude.common.util.QueueUtils.listFromQueue;

/**
 * 主键查询合并请求、结果拆分
 *
 * @author explore
 * @since 2021/11/29 17:57
 **/
public class QueueServiceImpl<M extends BaseMapper<T>, T extends Model<T>> extends ServiceImpl<M, T> {
    protected LinkedBlockingQueue<ModelWrapper<T>> queue = new LinkedBlockingQueue<>();
    
    /**
     * 子类初始化，每200毫秒发车一次，每次最多100个请求
     */
    @PostConstruct
    public void init() {
        ScheduledExecutorService service = new ScheduledThreadPoolExecutor(1);
        Runnable runnable = () -> {
            int size = Math.min(queue.size(), 100);
            if (size == 0) {
                return;
            }
            // 从队列中取出多少条记录
            Collection<ModelWrapper<T>> requests = listFromQueue(queue, size);
            // 将多个请求合并为一个请求
            Set<Serializable> ids = requests.stream().map(ModelWrapper::getId).collect(toSet());
            // 批量查询结果，转化为Map
            Map<Serializable, T> map = listByIds(ids).stream().collect(toMap(Model::pkVal, e -> e));
            // 将查询结果拆分并分发
            requests.forEach(e -> e.getFuture().complete(map.get(e.getId())));
        };
        // 每200毫秒发车一次
        service.scheduleAtFixedRate(runnable, 0, 200, TimeUnit.MILLISECONDS);
    }
    
    /**
     * 重写主键查询方法
     *
     * @param id 主键ID
     * @return 实体对象
     */
    @Override
    public T getById(Serializable id) {
        CompletableFuture<T> future = new CompletableFuture<>();
        queue.add(new ModelWrapper<>(id, future));
        try {
            return future.get();
        } catch (InterruptedException | ExecutionException e) {
            e.printStackTrace();
        }
        return null;
    }
}
