package zeiss.gesture.view;

import android.content.Context;
import android.graphics.Color;
import android.os.Handler;
import android.util.AttributeSet;
import android.util.Log;
import android.util.TypedValue;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.animation.AccelerateInterpolator;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.Scroller;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.List;

import zeiss.gesture.Util;
import zeiss.gesture.detector.ZeissGestureTouchDetector;


public class WheelView<T> extends ScrollView implements ZeissGestureTouchDetector.OnScrollListener {

    public interface OnWheelViewListener<T> {
        void onSelected(int selectedIndex, T item);

        void onScrollItemChanged(int previousIndex, int selectedIndex, T item, ArrayList<View> itemList);
    }

    public interface WheelViewInteface<T> {
        View onCreateItemView(T item, int index);
    }

    public static final String TAG = WheelView.class.getSimpleName();

    private static final int DIRECTION_UP = -1;
    private static final int DIRECTION_DOWN = 1;
    public enum SCROLL_DIRECTION{
        DIRECTION_UP,DIRECTION_DOWN
    }
    private SCROLL_DIRECTION direction = SCROLL_DIRECTION.DIRECTION_DOWN;
    int flingY;
    private int DEFAULT_ITEM_HEIGHT = 50;
    private int DEFAULT_DISPLAY_ITEM = 5;

    private Context context;

    private RelativeLayout relViews;
    private LinearLayout views;
    private View selectedView;

    private int layoutHeight, layoutWidth;

    int displayItemCount = DEFAULT_DISPLAY_ITEM;
    int selectedIndex = 0;
    private int previousSelectedIndex;
    int itemHeight = 0;
    //default font size 20
    private int textSize = 20;
    private ArrayList<View> itemList = new ArrayList<>();
    List<T> items;

    private OnWheelViewListener<T> onWheelViewListener;
    private WheelViewInteface<T> onWheelViewInteface;

    public WheelView(Context context) {
        super(context);
        init(context);
        
    }

    public WheelView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public WheelView(Context context, AttributeSet attrs, int defStyle) {
        super(context, attrs, defStyle);
        init(context);
    }

    @Deprecated
    public void createAllView() {
        initData();
    }

    public void commit() {
        initData();
    }

    public void setItems(List<T> list) {
        if (null == items) {
            items = new ArrayList<T>();
        }
        items.clear();
        items.addAll(list);
    }

    private void init(Context context) {
        this.context = context;
        this.setVerticalScrollBarEnabled(false);

        relViews = new RelativeLayout(context);
        addView(relViews);

        selectedView = new View(context);

        views = new LinearLayout(context);
        views.setOrientation(LinearLayout.VERTICAL);

        relViews.addView(selectedView);
        relViews.addView(views);

        //default color
        setSelectedViewColor(Color.parseColor("#CCFFFFFF"));

        onFlingRunnable = new OnFlingRunnable(context, new OnFlingRunnable.Callback() {
            @Override
            public void started() {
                Log("OnFlingRunnable -- started");
            }

            @Override
            public void update(int position) {
                Log("OnFlingRunnable -- update : " + position);
                scrollViewBy(position);
            }

            @Override
            public void stopped() {
                Log("OnFlingRunnable -- stopped");
                selectItem(selectedIndex);
                onSelectedCallBack();
            }
        });
    }

    private void initData() {
        for (int i = 0; i < items.size(); i++) {
            //put the item into array list
            itemList.add(createView(items.get(i), i));
            views.addView(itemList.get(i));
        }
        onScrollChanged(previousSelectedIndex, selectedIndex);
    }

    public int getTextSize() {
        return textSize;
    }

    public void setTextSize(int size) {
        textSize = size;
    }

    private View customView(T item, int index) {
        View view = null;
        if (onWheelViewInteface != null) {
            view = onWheelViewInteface.onCreateItemView(item, index);
        }
        if (view == null) {
            TextView tv = new TextView(context);
            tv.setSingleLine(true);
            tv.setTextSize(TypedValue.COMPLEX_UNIT_SP, textSize);
            tv.setText(item.toString());
            tv.setGravity(Gravity.LEFT);
            tv.setTag(index);
            tv.setTextColor(Color.LTGRAY);
            tv.setTextSize(TypedValue.COMPLEX_UNIT_SP, textSize);

            int padding = dip2px(4);
            tv.setPadding(padding, padding, padding, padding);
            tv.setLayoutParams(new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, getViewMeasuredHeight(tv)));
            view = tv;
        }
        return view;
    }

    private View createView(T item, int index) {

        View tv = customView(item, index);

        if (0 == itemHeight) {
            if (tv.getLayoutParams() != null && tv.getLayoutParams().height != 0) {
                itemHeight = tv.getLayoutParams().height;
            } else {
                itemHeight = DEFAULT_ITEM_HEIGHT;
            }
            relViews.setLayoutParams(new LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT));
            views.setLayoutParams(new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT));
            selectedView.setLayoutParams(new RelativeLayout.LayoutParams(RelativeLayout.LayoutParams.MATCH_PARENT, itemHeight));
        }
        return tv;
    }

    private void scrollViewBy(float y) {
        scrollViewTo((int)selectedView.getY()  - ((int) y * getDirection()));
    }

    private void scrollViewTo(int newPositionY) {

        float totalHeigh = getChildAt(0).getHeight();
        float centerY = layoutHeight / 2 - itemHeight / 2;
        float canMoveY = totalHeigh - layoutHeight;
        float lastPositionY = canMoveY + centerY;

        smoothScrollTo(0, newPositionY);

        if (newPositionY > lastPositionY) {
            smoothScrollTo(0, (int) canMoveY);
        } else if (newPositionY <= centerY) {
            smoothScrollTo(0, 0);
        } else {
            smoothScrollTo(0, (int) (newPositionY - centerY));
        }

        updateSelectedView(newPositionY);
    }

    private void updateSelectedView(int newPositionY) {

        float positionHeight = 0;
        float totalHeigh = getChildAt(0).getHeight();
        float centerY = layoutHeight / 2 - itemHeight / 2;
        float canMoveY = totalHeigh - layoutHeight;
        float lastPositionY = canMoveY + centerY;
//        Log("updateSelectedView totalHeigh  = "+totalHeigh +" centerY = "+centerY+" canMoveY = "+canMoveY+" lastPositionY = "+lastPositionY);
        if (newPositionY > lastPositionY) {
            if (newPositionY >= totalHeigh - itemHeight) {
                selectedView.setY(totalHeigh - itemHeight);
            } else {
                selectedView.setY(newPositionY);
            }
            positionHeight = selectedView.getY() + itemHeight / 2;
        } else if (newPositionY <= centerY) {
            if (newPositionY < 0) {
                selectedView.setY(0);
            } else {
                selectedView.setY(newPositionY);
            }
            positionHeight = selectedView.getY() + itemHeight / 2;
        } else {
            selectedView.setY(newPositionY);
            positionHeight = centerY + itemHeight / 2 + getScrollY();
        }
        int index = (int) (positionHeight / itemHeight);
        previousSelectedIndex = selectedIndex;
        if(index != selectedIndex){
            onScrollChanged(previousSelectedIndex, index);
        }
        selectedIndex = index;
//        Log("liam = "+selectedIndex +" positionHeight = "+positionHeight+" itemHeight = "+itemHeight);
    }

    public int getSelectedIndex(){
        return selectedIndex;
    }

    public void selectItem(final int position) {
        onScrollChanged(previousSelectedIndex, selectedIndex);
        if (selectedView != null) {
            selectedView.post(new Runnable() {
                @Override
                public void run() {
                    scrollViewTo(itemHeight * position);
                }
            });
        }
    }

    private void onSelectedCallBack() {
        if (null != onWheelViewListener) {
            onWheelViewListener.onSelected(selectedIndex, items.get(selectedIndex));
        }
    }

    private void onScrollChanged(int previousSelectedIndex, int index) {
        onWheelViewListener.onScrollItemChanged(previousSelectedIndex, index, items.get(selectedIndex), itemList);
    }

//    @Override
//    public void fling(int velocityY) {
//        super.fling(velocityY/3);
//    }

    public void setOnWheelViewListener(OnWheelViewListener<T> onWheelViewListener) {
        this.onWheelViewListener = onWheelViewListener;
    }

    public void setOnWheelViewInterface(WheelViewInteface<T> onWheelViewInteface) {
        this.onWheelViewInteface = onWheelViewInteface;
    }

    private int dip2px(float dpValue) {
        final float scale = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * scale + 0.5f);
    }

    private int getViewMeasuredHeight(View view) {
        int width = MeasureSpec.makeMeasureSpec(0, MeasureSpec.UNSPECIFIED);
        int expandSpec = MeasureSpec.makeMeasureSpec(Integer.MAX_VALUE >> 2, MeasureSpec.AT_MOST);
        view.measure(width, expandSpec);
        return view.getMeasuredHeight();
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);
        setMeasuredDimension(MeasureSpec.getSize(widthMeasureSpec), MeasureSpec.getSize(heightMeasureSpec));

        int newLayoutHeight = MeasureSpec.getSize(heightMeasureSpec);
        int newLayoutWidth = MeasureSpec.getSize(widthMeasureSpec);

        if (newLayoutHeight != layoutHeight || newLayoutWidth != layoutWidth) {
            layoutHeight = newLayoutHeight;
            layoutWidth = newLayoutWidth;
            onMeasureUpdateLayoutSize();
        }
    }

    private void onMeasureUpdateLayoutSize() {
        int displayHeight = itemHeight * displayItemCount;
        if (displayHeight > layoutHeight) {
            displayHeight = layoutHeight;
        }
        LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams) this.getLayoutParams();
        setLayoutParams(new LinearLayout.LayoutParams(lp.width, displayHeight));
    }


    public void setSelectedViewColor(int color) {
        selectedView.setBackgroundColor(color);
    }

    public void setBackgroundResource(int id){
        selectedView.setBackgroundResource(id);
    }

    public void setDisplayItemCount(int displayItemCount) {
        this.displayItemCount = displayItemCount;
    }

    public void scrollToTop() {
        selectItem(0);
    }

    public void scrollToBottom() {
        selectItem(items.size());
    }

    public T getSeletedItem() {
        return items.get(selectedIndex);
    }

    private void Log(String msg) {
        if(Util.DEBUG) {
            Log.i(TAG, msg);
        }
    }


    @Override
    public void onStart(MotionEvent event) {

    }

    @Override
    public void onMove(MotionEvent e1, MotionEvent e2, float distanceX, float distanceY, boolean isDrag) {
        if (isDrag && distanceY != 0) {
            scrollViewBy(distanceY);
        }
    }

    @Override
    public void onMoveEnd(MotionEvent event) {
        selectItem(selectedIndex);
        onSelectedCallBack();
//        Log("AAA onMoveEnd");
    }

    @Override
    public void onFling(MotionEvent e1, MotionEvent e2, float velocityX, float velocityY) {
//        Log("AAA onFling "+velocityY);
        flingY = (int) (velocityY * -1) / 2;
        Log("onFling OnFlingRunnable = "+flingY+" selectedIndex = "+selectedIndex +" previousSelectedIndex = "+previousSelectedIndex);

        onFlingRunnable.onFling((int) (velocityY * -1) / 2);
        onFlingRunnable.start();
    }


    private OnFlingRunnable onFlingRunnable;

    public static class OnFlingRunnable implements Runnable {
        interface Callback {
            void started();

            void update(int position);

            void stopped();
        }

        private static final int DELAY_MILLIS = 0;
        private Callback mCallback;
        private int mPreviousPosition;
        private Scroller mScroller;
        private Handler handler;
        private boolean isStart;

        public boolean isStart() {
            return isStart;
        }

        public OnFlingRunnable(Context context, Callback callback) {
            handler = new Handler();
            mCallback = callback;
            mScroller = new Scroller(context, new AccelerateInterpolator());
            isStart = false;
        }

        void start() {
            Log.e("AAA", "start OnFlingRunnable");
            isStart = true;
            mPreviousPosition = 0;
            handler.postDelayed(this, DELAY_MILLIS);
        }

        void stop() {
            mPreviousPosition = 0;
            mCallback.stopped();
            handler.removeCallbacks(this);
            isStart = false;
        }

        void onFling(int y) {
            Log.e("AAA", "onFling OnFlingRunnable -- y = " + (y));
            mScroller.fling(0, 0, 0, y,
                    Integer.MIN_VALUE,
                    Integer.MAX_VALUE,
                    Integer.MIN_VALUE,
                    Integer.MAX_VALUE);

            Log.e("AAA", "OnFlingRunnable -- onFling : getCurrY = " + mScroller.getCurrY() + " getFinalY = " + mScroller.getFinalY() + " getStartY = " + mScroller.getStartY());
        }


        @Override
        public void run() {
            mScroller.computeScrollOffset();
            int position = mScroller.getCurrY();
            if (mPreviousPosition - position == 0) {
                stop();
            } else {
                if (mPreviousPosition == 0) {
                    mCallback.started();
                } else {
//                        Log.e("AAA","OnFlingRunnable -- diff = "+(position - mPreviousPosition));
                    mCallback.update(position - mPreviousPosition);
                }
                mPreviousPosition = mScroller.getCurrY();
                handler.postDelayed(this, DELAY_MILLIS);
            }
        }
    }


    public void onResume() {

    }


    public void onStop() {
        if (onFlingRunnable != null && onFlingRunnable.isStart()) {
            onFlingRunnable.stop();
        }
    }


    private int getDirection(){
        switch (direction) {
            case DIRECTION_UP:
                return DIRECTION_UP;
            case DIRECTION_DOWN:
                return DIRECTION_DOWN;
        }
        return DIRECTION_UP;
    }

}
